/**
  ******************************************************************************
  * @file    main.c
  * @author  MCU Application Team
  * @brief   Main program body
  ******************************************************************************
  * @attention
  *
  * <h2><center>&copy; Copyright (c) 2023 Puya Semiconductor Co.
  * All rights reserved.</center></h2>
  *
  * This software component is licensed by Puya under BSD 3-Clause license,
  * the "License"; You may not use this file except in compliance with the
  * License. You may obtain a copy of the License at:
  *                        opensource.org/licenses/BSD-3-Clause
  *
  ******************************************************************************
  * @attention
  *
  * <h2><center>&copy; Copyright (c) 2016 STMicroelectronics.
  * All rights reserved.</center></h2>
  *
  * This software component is licensed by ST under BSD 3-Clause license,
  * the "License"; You may not use this file except in compliance with the
  * License. You may obtain a copy of the License at:
  *                        opensource.org/licenses/BSD-3-Clause
  *
  ******************************************************************************
  */

/* Includes ------------------------------------------------------------------*/
#include "main.h"
#include "py32t090xx_ll_Start_Kit.h"

/* Private define ------------------------------------------------------------*/
/* Private variables ---------------------------------------------------------*/
uint32_t LED_BlinkingTime = 0;
LL_GPIO_InitTypeDef GPIO_InitStruct = {0};

/* Private user code ---------------------------------------------------------*/
/* Private macro -------------------------------------------------------------*/
/* Private function prototypes -----------------------------------------------*/
static void APP_SystemClockConfig(void);
static void APP_EnterStandby(void);

/**
  * @brief  Main program.
  * @param  None
  * @retval int
  */
int main(void)
{
  /* Enable SYSCFG and PWR clock */
  LL_APB1_GRP2_EnableClock(LL_APB1_GRP2_PERIPH_SYSCFG);
  LL_APB1_GRP1_EnableClock(LL_APB1_GRP1_PERIPH_PWR);

  /* Configure system clock */
  APP_SystemClockConfig();

  /* Initialize LED */
  BSP_LED_Init(LED_GREEN);

  /* Initialize BUTTON */
  BSP_PB_Init(BUTTON_KEY,BUTTON_MODE_GPIO);

  /* Turn on the LED */
  BSP_LED_On(LED_GREEN);
  
  if (LL_PWR_IsActiveFlag_SB() != 0)
  {
    /* Clear the STANDBY flag */
    LL_PWR_ClearFlag_SB();

    /* Clear the wake-up flag */
    LL_PWR_ClearFlag_WU();
    
    LED_BlinkingTime=0;
    
    while(LED_BlinkingTime<10)
    {
      LL_mDelay(500);
      BSP_LED_Toggle(LED_GREEN);
      LED_BlinkingTime++;
    }    
  }

  /* Enable pull-up and pull-down configuration */
  LL_PWR_EnablePUPDCfg();
  
  /* Before entering standby mode, the pull-up function needs to be enabled 
     when P13 is used as the NRST_Pin, whereas it does not need to be enabled 
     when PD13 is used as a GPIO */
  LL_PWR_EnableGPIOPullUp(LL_PWR_GPIO_D, LL_PWR_GPIO_BIT_13);
  
  /* The Following Wakeup sequence is highly recommended prior to each Standby mode entry
     mainly when using more than one wakeup source this is to not miss any wakeup event.
     - Configure GPIO,
     - Disable all used wakeup sources,
     - Clear all related wakeup flags,
     - Re-enable all used wakeup sources,
     - Enter the Standby mode.
  */
  /* Configure PA1 as input mode */
  LL_IOP_GRP1_EnableClock(LL_IOP_GRP1_PERIPH_GPIOA);
  GPIO_InitStruct.Mode   = LL_GPIO_MODE_INPUT;
  GPIO_InitStruct.Pull   = LL_GPIO_PULL_DOWN;
  GPIO_InitStruct.Speed  = LL_GPIO_SPEED_FREQ_VERY_HIGH;
  GPIO_InitStruct.Pin    = LL_GPIO_PIN_1;
  LL_GPIO_Init(GPIOA, &GPIO_InitStruct);
  
  /* Disable all used wakeup sources: PWR_WAKEUP_PIN4 */
  LL_PWR_DisableWakeUpPin(LL_PWR_WAKEUP_PIN4);
  
  /* Clear the wake-up flag */
  LL_PWR_ClearFlag_WU();
  
  /* Set the Wake-Up pin polarity high for the event detection */
  LL_PWR_SetWakeUpPinPolarityHigh(LL_PWR_WAKEUP_PIN4);
  
  /* Enable WakeUp Pin PWR_WAKEUP_PIN4 connected to PA1 */
  LL_PWR_EnableWakeUpPin(LL_PWR_WAKEUP_PIN4);
  
  /* Wait for the button to be pressed */
  while(BSP_PB_GetState(BUTTON_KEY) == 0);

  /* Turn off the LED */
  BSP_LED_Off(LED_GREEN);

  /* Enter STANDBY mode */
  APP_EnterStandby();

  while (1)
  {
    /* LED toggle */
    BSP_LED_Toggle(LED_GREEN);
    LL_mDelay(500);
  }
}

/**
  * @brief  Configure system clock
  * @param  None
  * @retval None
  */
static void APP_SystemClockConfig(void)
{
  /* Enable HSI */
  LL_RCC_HSI_Enable();
  while(LL_RCC_HSI_IsReady() != 1)
  {
  }

  /* Set AHB prescaler: HCLK = SYSCLK */
  LL_RCC_SetAHBPrescaler(LL_RCC_SYSCLK_DIV_1);

  /* Select HSISYS as system clock source */
  LL_RCC_SetSysClkSource(LL_RCC_SYS_CLKSOURCE_HSISYS);
  while(LL_RCC_GetSysClkSource() != LL_RCC_SYS_CLKSOURCE_STATUS_HSISYS)
  {
  }

  /* Set APB prescaler: PCLK = HCLK */
  LL_RCC_SetAPB1Prescaler(LL_RCC_APB1_DIV_1);
  LL_Init1msTick(8000000);

  /* Update the SystemCoreClock global variable(which can be updated also through SystemCoreClockUpdate function) */
  LL_SetSystemCoreClock(8000000);
}

static void APP_EnterStandby(void)
{
  /* Clear the wake-up flag */
  LL_PWR_ClearFlag_WU();
  
  /* Set Low-Power mode */
  LL_PWR_SetPowerMode(LL_PWR_MODE_STANDBY);
  
  /* Voltage Regulator selects DLPR mode */
  LL_PWR_SetVoltageRegulatorMode(LL_PWR_VOLTAGE_REGULATOR_MODE_DLPR);
  
  /* Set SLEEPDEEP bit of Cortex System Control Register */
  LL_LPM_EnableDeepSleep();
  
  /* This option is used to ensure that store operations are completed */
#if defined ( __CC_ARM)
  __force_stores();
#endif
  /* Request Wait For Interrupt */
  __WFI();
}

/**
  * @brief  Error executing function.
  * @param  None
  * @retval None
  */
void APP_ErrorHandler(void)
{
  while (1)
  {
  }
}

#ifdef  USE_FULL_ASSERT
/**
  * @brief  Reports the name of the source file and the source line number
  *         where the assert_param error has occurred.
  * @param  file: pointer to the source file name
  * @param  line: assert_param error line source number
  * @retval None
  */
void assert_failed(uint8_t *file, uint32_t line)
{
  /* Users can add their own printing information as needed,
     for example: printf("Wrong parameters value: file %s on line %d\r\n", file, line) */
  /* Infinite loop */
  while (1)
  {
  }
}
#endif /* USE_FULL_ASSERT */

/************************ (C) COPYRIGHT Puya *****END OF FILE******************/
